;    Virtual memory read/write code for ZXZVM: Z-Code interpreter for the Z80 processor
;    Copyright (C) 2019  Kevin Thacker<kev@cpcfreak.co.uk>
;
;    This program is free software; you can redistribute it and/or modify
;    it under the terms of the GNU General Public License as published by
;    the Free Software Foundation; either version 2 of the License, or
;    (at your option) any later version.
;
;    This program is distributed in the hope that it will be useful,
;    but WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    GNU General Public License for more details.
;
;    You should have received a copy of the GNU General Public License
;    along with this program; if not, write to the Free Software
;    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; NOTE: None of this has been optimised

;;------------------------------------------------
;; Read from Z-Machine RAM or ROM
;; Enter:
;; DHL = virtual address
;; IX = page table
;; Exit:
;; A = byte, other registers corrupted
read_virtual:
push ix
call store_current_virt		;; store virtual memory address being accessed

call map_virtual_to_physical	;; is the virtual page mapped into physical ram?
jr nc,rv2

;; no, we need to map it.
pop ix
push ix
call cache_virtual

rv2:
pop ix
ld a,(hl)
ret

;;------------------------------------------------
;; Write to Z-Machine RAM

;; Enter:
;; IX = page table
;; DHL = virtual address
;; A = byte
;; Exit:
;; registers corrupted.
write_virtual:

ld (virtual_data),a

ld a,d			;; do not allow write above 64KB
or a
ret nz

push ix
call store_current_virt			;; store virtual memory address being accessed

call map_virtual_to_physical		;; is the virtual page mapped into physical ram?
jr nc,wv2

;; no, we need to map it.
pop ix
push ix
call cache_virtual

wv2:
call set_page_dirty				;; mark as modified - will need commiting if discarded
pop ix
ld a,(virtual_data)
ld (hl),a
ret


;;------------------------------------------------
;; Store virtual address
;; Entry:
;; DHL = virtual address
store_current_virt:
ld (virtual_address+0),hl
ld a,d
ld (virtual_address+2),a
ret

;;------------------------------------------------
;; Get virtual address

;; Exit:
;; DHL = virtual address
get_current_virt:
ld hl,(virtual_address+0)
ld a,(virtual_address+2)
ld d,a
ret

;;------------------------------------------------
;; Discard page, find an old page and load a new page in

cache_virtual:
call get_page_index
call get_page_addr

;; found an old page so commit existing one in the same page table location
call commit_page

;; set new virtual address for this page
call allocate_page

;; read virtual memory into physical
call restore_virtual_page

;; map virtual to physical
call virtual_to_physical
or a
ret
;;------------------------------------------------

set_page_dirty:
ld a,1
ld (ix+page_flags),a
ret

;;------------------------------------------------
;; Setup page data which defines the physical addresses available
;; and initial corresponding virtual pages

;; Enter:
;; HL = address of physical ram start to use for paging virtual memory into
;; DE = length of physical ram 
;; IX = page table
;; Will allocate pages for virtual memory use.
virt_init:
;; count number of pages we can cache at one time
push hl




ex de,hl
;; enough space to fit the pages?
ld bc,PAGE_SIZE_BYTES*VIRT_NUM_PAGES
or a
sbc hl,bc
ld a,h
bit 7,a
jr nz,vinit3
ld (ix+0),VIRT_NUM_PAGES
jr vinit2

vinit3:
;; failed
scf
ret

vinit2:
pop hl

ld b,(ix+0)
inc b
dec b
ret z
push ix
inc ix
vinit1:
;; set physical page address
ld (ix+page_physical_address+0),l
ld (ix+page_physical_address+1),h
;; clear modified
ld (ix+page_flags),0
;; set initial age
ld de,PAGE_SIZE_BYTES
add hl,de
ld de,page_data_size
add ix,de
djnz vinit1
pop ix
or a
ret


;;------------------------------------------------
;; for game restart, setup default virtual addresses and 
;; read pages in for those
;; Entry:
;; IX = page table

virt_reinit:
call virt_set_default
call virt_restore_init
ret

;;------------------------------------------------
;; set default virtual addresses - mapped to consecutive pages in z-machine ram

;; Entry:
;; IX = page table
virt_set_default:
ld b,(ix+0)
inc b
dec b
ret z

push ix
;; set virtual page address
ld hl,0
ld d,0
inc ix
vinit4:
push bc
ld (ix+page_virtual_address+0),l
ld (ix+page_virtual_address+1),h
ld (ix+page_virtual_address+2),d
ld bc,PAGE_SIZE_BYTES
add hl,bc
ld bc,page_data_size
add ix,bc
pop bc
djnz vinit4
pop ix
ret


;;------------------------------------------------
;; read initial virtual pages
;; Entry:
;; IX = page table

virt_restore_init:
ld b,(ix+0)
inc b
dec b
ret z
push ix
inc ix
vinit5:
push bc
call restore_virtual_page
ld bc,page_data_size
add ix,bc
pop bc
djnz vinit5
pop ix
ret


;;------------------------------------------------
;; If page is modified commit it back to swap
;;
;; IX = current page table entry

commit_page:
ld a,(ix+page_flags)
or a
ret z
xor a
ld (ix+page_flags),a
;; page is dirty
call store_virtual_page
ret



;;------------------------------------------------
;; get address of page table entry from index
;;
;; Entry:
;; A = index in page table
;; IX = start of page table
;;
;; Exit:
;; IX = address in page table
get_page_addr:
push hl
push de
ld l,a
ld h,0
add hl,hl	;; x2
ld e,l
ld d,h
add hl,hl	;; x4
add hl,de	;; x6
;; 6 is size of a page table entry.
ld e,l
ld d,h
add ix,de
pop de
pop hl
ret



;;------------------------------------------------
;; Calculate index in page table based on virtual address
;; Entry:
;; DHL = virtual address
get_page_index:
;; first shift down dividing by the size of the page size in bytes
if PAGE_SIZE_BYTES=128
rept 7
srl d 
rr h
rr l
endm
endif

if PAGE_SIZE_BYTES=256
rept 8
srl d 
rr h
rr l
endm
endif

if PAGE_SIZE_BYTES=1024
rept 9
srl d 
rr h
rr l
endm
endif

if PAGE_SIZE_BYTES=2048
rept 10
srl d 
rr h
rr l
endm
endif

if PAGE_SIZE_BYTES=4096
rept 11
srl d 
rr h
rr l
endm
endif

if PAGE_SIZE_BYTES=8192
rept 12
srl d 
rr h
rr l
endm
endif

if PAGE_SIZE_BYTES=16384
rept 13
srl d 
rr h
rr l
endm
endif

;; now AND to get index in table, values then wrap so that
;; multiple virtual pages map onto the same entry in the page table
ld a,l
if VIRT_NUM_PAGES=2
and &1
endif

if VIRT_NUM_PAGES=4
and &3
endif

if VIRT_NUM_PAGES=8
and &7
endif

if VIRT_NUM_PAGES=16
and &f
endif

if VIRT_NUM_PAGES=32
and &1f
endif

if VIRT_NUM_PAGES=64
and &3f
endif

if VIRT_NUM_PAGES=128
and &7f
endif

if VIRT_NUM_PAGES=256
endif

ret

;;------------------------------------------------
;; Page align virtual address
;; Entry:
;; DHL = virtual address
;; Exit:
;; DHL = page aligned virtual address
align_virtual_page:
if PAGE_SIZE_BYTES=128
ld a,l
and %10000000
ld l,a
endif

if PAGE_SIZE_BYTES=256
ld l,0
endif

if PAGE_SIZE_BYTES=512
ld l,0
ld a,h
and &fe
ld h,a
endif

if PAGE_SIZE_BYTES=1024
ld l,0
ld a,h
and &fc
ld h,a
endif

if PAGE_SIZE_BYTES=2048
ld l,0
ld a,h
and &f8
ld h,a
endif

if PAGE_SIZE_BYTES=4096
ld l,0
ld a,h
and &f0
ld h,a
endif

if PAGE_SIZE_BYTES=8192
ld l,0
ld a,h
and &e0
ld h,a
endif

if PAGE_SIZE_BYTES=16384
ld l,0
ld a,h
and &c0
ld h,a
endif

ret

;;------------------------------------------------
;; allocate a page
;;
;; Enter:
;; ix = page table entry
;; dhl = virtual address (unaligned)
allocate_page:
call get_current_virt
call align_virtual_page
ld (ix+page_virtual_address+0),l
ld (ix+page_virtual_address+1),h
ld (ix+page_virtual_address+2),d
ret

;;------------------------------------------------
;; Find the physical address for a virtual page (if mapped)
;; This is slow linear search.
;;
;; Input:
;; DHL = virtual address
;; IX = page table
;; Output:
;; if virtual page is mapped: carry clear. HL = physical address, IX = page data address
;; if virtual page is not mapped: carry false. Registers corrupt.
map_virtual_to_physical:
;; based on virtual address map to index
call get_page_index
;; get pointer to page
call get_page_addr
;; get current
call get_current_virt
;; align
call align_virtual_page
cp (ix+page_virtual_address+0)
jr nz,mvtop2	;; virtual page we want not mapped here
ld a,h
cp (ix+page_virtual_address+1)
jr nz,mvtop2 ;; virtual page we want not mapped here
ld a,d
cp (ix+page_virtual_address+2)
jr z,mvtop3	;; virtual page we want is mapped here

mvtop2:
;; virtual page we want not mapped here
scf
ret

mvtop3:

;; success map virtual to physical
call virtual_to_physical
or a
ret

;;---------------------------------------------------------------------
;; Map virtual to physical
;; Entry:
;; IX = current page table entry
;; Exit:
;; HL = address in physical z80 ram corresponding to virtual address 
virtual_to_physical:
call get_current_virt	;; unaligned
;; calc relative to aligned virtual page address
ld a,d
sub (ix+page_virtual_address+2)
ld d,a
ld a,h
sbc a,(ix+page_virtual_address+1)
ld h,a
ld a,l
sbc a,(ix+page_virtual_address+0)
ld l,a
call get_page_physical_address
;; add on location in physical memory
add hl,de
ret

;;---------------------------------------------------------------------
;; IX = current page table entry
get_page_physical_address:
ld e,(ix+page_physical_address+0)
ld d,(ix+page_physical_address+1)
ret
;;---------------------------------------------------------------------
;; temp for storing data
virtual_data:
defb 0
virtual_address:
defs 3
