;    Swap file read/write code for ZXZVM: Z-Code interpreter for the Z80 processor
;    Copyright (C) 2019  Kevin Thacker<kev@cpcfreak.co.uk>
;
;    This program is free software; you can redistribute it and/or modify
;    it under the terms of the GNU General Public License as published by
;    the Free Software Foundation; either version 2 of the License, or
;    (at your option) any later version.
;
;    This program is distributed in the hope that it will be useful,
;    but WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    GNU General Public License for more details.
;
;    You should have received a copy of the GNU General Public License
;    along with this program; if not, write to the Free Software
;    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;------------------------------------------------
CPM_RECORD_SIZE equ 128

;; size of z-machine ram
ZMACHINE_RAM_SIZE equ 65536

;; number of records in swap corresponds to 64KB of z-machine ram
MAX_RECORDS_ZMACHINE_RAM equ 512

;;------------------------------------------------
bdos_swap:
push iy
call 0005
pop iy
ret


;;------------------------------------------------
;; Convert Z-machine memory address to record number
;; within swap.dat file.
;; Entry:
;; IX = current page table entry
;; Exit:
;; DHL = record number
get_record_index:
ld l,(ix+page_virtual_address+0)
ld h,(ix+page_virtual_address+1)
ld d,(ix+page_virtual_address+2)
srl d ;; /2
rr h
rr l
srl d ;; /4
rr h
rr l
srl d ;; /8
rr h
rr l
srl d ;; /16
rr h
rr l
srl d ;; /32
rr h
rr l
srl d ;; /64
rr h
rr l
srl d ;; /128
rr h
rr l
ret

swap_exit:
or a
jp ZXEXIT

swap_write_exit:
ld hl,fail_write
jr swap_exit

swap_read_exit:
ld hl,fail_read
jr swap_exit

fail_write:
defb "Failed to write to swa","p"+&80

fail_read:
defb "Failed to read from swa","p"+&80

;;------------------------------------------------
;; Store Z-machine virtual memory page to swap.dat
;;
;; Entry:
;; IX = current page table entry
store_virtual_page:
;; ensure swap file has been opened
ld a,(swap_opened)
or a
ret z

push ix
;; if outside 64kb range then ignore.
;; Z-Machine has 64KB RAM
ld a,(ix+page_virtual_address+2)
or a
jr  nz,svp2

;; setup FCB random access page index
call get_record_index
ld a,l
ld (swap_fcb+&21),a
ld a,h
ld (swap_fcb+&22),a
ld a,d
ld (swap_fcb+&23),a


;; get physical ram address of virtual page
;; and set dma address
call get_page_physical_address

ld b,PAGE_SIZE_BYTES/CPM_RECORD_SIZE
svp1:
push bc
push de

;; set dma address
ld c,&1a
call bdos_swap

;; write random
ld de,swap_fcb
ld c,&22 
call bdos_swap
or a
jp nz,swap_write_exit

ld ix,swap_fcb
call inc_random

pop hl
ld bc,CPM_RECORD_SIZE
add hl,bc
ex de,hl
pop bc
djnz svp1


;; TODO: Do I need to do this? This is meant to ensure data is committed.
; set F5.
ld hl,swap_fcb
ld bc,5
add hl,bc
set 7,h
;; parial close
ld de,swap_fcb
ld c,&10
call bdos_swap
cp 255
jp z,swap_write_exit

svp2:
pop ix
ret


;;------------------------------------------------
;; IX = fcb
inc_random:
ld a,(ix+&21)
add a,1
ld (ix+&21),a
ld a,(ix+&22)
adc a,0
ld (ix+&22),a
ld a,(ix+&23)
adc a,0
ld (ix+&23),a
ret

;;------------------------------------------------
;; Read from swap.dat (if z-machine ram)
;; OR story file if z-machine ROM.
;; and read into physical memory
;; 
;; Entry:
;; IX = current page table entry
restore_virtual_page:
;; do nothing if swap has not been opened
ld a,(swap_opened)
or a
ret z

push ix
;; read from swap or from file?
ld a,(ix+page_virtual_address+2)
or a
ld de,swap_fcb	 ;; read from Z-Machine RAM if address<65536
jr z,rvp2
;; read from source (Z-Machine ROM)
ld de,(source_fcb)
rvp2:
ld (read_fcb),de

call get_record_index

push ix
;; dhl
ld ix,(read_fcb)
ld a,l
ld (ix+&21),a
ld a,h
ld (ix+&22),a
ld a,d
ld (ix+&23),a

pop ix

;; get physical ram address and set dma address
call get_page_physical_address

ld b,PAGE_SIZE_BYTES/CPM_RECORD_SIZE
rvp1:
push bc
push de

;; set dma address
ld c,&1a
call bdos_swap

;; read random
ld de,(read_fcb)    ;; which fcb to read from
ld c,&21 
call bdos_swap
or a
jp nz,swap_read_exit

ld ix,(read_fcb)
call inc_random

pop hl
ld bc,CPM_RECORD_SIZE
add hl,bc
ex de,hl
pop bc
djnz rvp1

pop ix
ret

;;------------------------------------------------
;; For restart, this will close the swap file
;; and re-generate the swap from the opened
;; story file

swap_reinit:
call swap_finish
call swap_initreinit
ret

;;------------------------------------------------
;; Generate swap.dat in it's entirety from the story file.
;; This caches as much as possible into a given buffer to reduce
;; seek times and deblocking.

;; TODO: Can this be made more optimal?
;; 


;; Entry:
;; BC = story file
;; HL = start of buffer 
;; DE = length of buffer
swap_init:
ld (source_fcb),bc
ld (buffer),hl

;; work out the number of CPM records that will fit into the buffer
ld l,e
ld h,d
ld d,0

ld bc,PAGE_SIZE_BYTES
swapi00a:
or a
sbc hl,bc
;; zero
ld a,h
or l
jr z,swapi00c

;; HL<DE?
bit 7,h
jr nz,swapi00c
inc d
jr swapi00a

swapi00c:
;; this is the max we can read at once
ld a,d
ld (num_records_init),a

swap_initreinit:
push de
push bc
push af
push ix

ld hl,swap_filename
ld b,end_swap_filename-swap_filename
call fill_in_fcb

;; for cpm2.2, must ensure the disc has been set to read/write
;; do the following in your startup code
;;ld c,&19			;; get current disk
;;call bdos_swap
;;push af
;;ld c,&0d			;; reset disk system
;;call bdos_swap
;;pop af
;;ld e,a
;;ld c,&0e			;; select disk restoring it to read/write state
;;call bdos_swap

ld c,&11
ld de,swap_fcb
call bdos_swap
cp 255
jr z,swapi0

;; delete swap.dat if it exists.
ld c,&13
ld de,swap_fcb
call bdos_swap

swapi0:

ld hl,swap_filename
ld b,end_swap_filename-swap_filename
call fill_in_fcb

;; now create new swap file
ld de,swap_fcb
ld c,&16
call bdos_swap
cp 255
jp z,swap_write_exit

ld a,1
ld (swap_opened),a

;; set cr to 0 for sequential write
ld hl,swap_fcb+&20
ld (hl),0

;; get length of story file in records
ld de,(source_fcb)
ld c,&23
call bdos_swap


ld hl,(source_fcb)
ld de,&21
add hl,de
ld e,(hl)
inc hl
ld d,(hl)
inc hl
ld a,(hl)
ld l,e
ld h,d
ld d,a

;; cpm 2.2
;; if d=1 then size exceeds max record count of 65536 ~8MB
ld a,d
or a
jp nz,swap_write_exit

;; 512 is 65536/128 = number of records in 64KB
;; and is the max size for the swap.dat
ld bc,MAX_RECORDS_ZMACHINE_RAM	

;; limit to max 64KB in records
ld a,d
or a
jr nz,h2
ld a,h
cp MAX_RECORDS_ZMACHINE_RAM/256		
jr nc,h2
ld c,l
ld b,h
h2:
push bc
push bc

;; reset random record for story to read from the start
ld hl,(source_fcb)
ld de,&21
add hl,de
ld (hl),0
inc hl
ld (hl),0
inc hl
ld (hl),0

pop bc
;; if size of file was 0, then don't read
ld a,b
or c
jr z,swapi0a

;;-------------------------------------------------------------------------------------------------------------
;; main cache loop

swapi1:

;; B is non-zero > therefore BC>num_records_init
ld a,b
or a
ld a,(num_records_init)
jr nz,swapi1b
;; A<C -> therefore C>num_records_init
cp c
jr c,swapi1b

;; A>C - therefore read a records
ld a,c

swapi1b:
;; A = number of records to read this pass
ld (init_records_read),a

;;---------------------------------------------------------------------------------------------------------------
;; read records from story

ld de,(buffer)
ld a,(init_records_read)
swapi1c:
push af
push bc
push de


ld c,&1a   ;; disk transfer address/set dma address
call bdos_swap

ld de,(source_fcb)
ld c,&21 ;; read random
call bdos_swap
or a
jp nz,swap_write_exit

;; increment random record number
ld ix,(source_fcb)
call inc_random

pop hl
ld bc,CPM_RECORD_SIZE
add hl,bc
ex de,hl
pop bc

dec bc
pop af
dec a
jr nz,swapi1c

;;--------------------------------------------------------------------------------------------------
;; write records from buffer

push bc ;; current count of records remaining

ld de,(buffer)
ld a,(init_records_read)
swapi1d:
push af
push de

ld c,&1a   ;; disk transfer address/set dma address
call bdos_swap

ld de,swap_fcb
ld c,&15 ;; sequential write
call bdos_swap
or a
jp nz,swap_write_exit

pop hl
ld bc,CPM_RECORD_SIZE
add hl,bc
ex de,hl
pop af
dec a
jr nz,swapi1d
pop bc

ld a,b
or c
jp nz,swapi1

swapi0a:
pop bc
ld hl,MAX_RECORDS_ZMACHINE_RAM
or a
sbc hl,bc
ld a,h
or l
jr z,swapi1e
ld c,l
ld b,h

;; if story was less than 64KB fill up swap.dat to 64KB.

push hl
push de
push bc
;; clear record sized part of buffer
ld hl,(buffer)
ld e,l
ld d,h
inc de
ld (hl),0
ld bc,CPM_RECORD_SIZE
ldir
pop bc
pop de
pop hl

swapi1a:
push bc

ld c,&1a   ;; disk transfer address/set dma address
ld de,(buffer)
call bdos_swap

ld de,swap_fcb
ld c,&15 ;; sequential write
call bdos_swap
or a
jp nz,swap_write_exit

pop bc
dec bc		
ld a,b
or c
jr nz,swapi1a

swapi1e:

;; TODO: Do I need to do this? This is meant to ensure data is committed.
;; set F5.
ld hl,swap_fcb
ld bc,5
add hl,bc
set 7,h
;; parial close
ld de,swap_fcb
ld c,&10
call bdos_swap
cp 255
jp z,swap_write_exit


pop ix
pop af
pop bc
pop de
scf
ret

;;------------------------------------------------
;; Close swap file properly.
;;
;;
swap_finish:
ld a,(swap_opened)
or a
ret z

;; this is now a full close

;; clear F5
ld hl,swap_fcb
ld bc,5
add hl,bc
res 7,h

;; close swap fcb
ld de,swap_fcb
ld c,&10
call bdos_swap
;; do not handle error here

xor a
ld (swap_opened),a
ret


;;------------------------------------------------
;; initialise fcb - clear it and fill in filename
fill_in_fcb:
push bc
push hl
ld hl,swap_fcb
ld e,l
ld d,h
inc de
ld bc,end_swap_fcb-swap_fcb-1
ld (hl),0
ldir
ld hl,swap_fcb+1
ld e,l
ld d,h
inc de
ld bc,11-1
ld (hl),' '
ldir
pop hl
pop bc
ld de,swap_fcb+1
fn_name:
ld a,(hl)
inc hl
cp '.'
jr z,fn2
ld (de),a
inc de
djnz fn_name
jr fn4
fn2:
dec b
ld de,swap_fcb+9
fn_ext:
ld a,(hl)
ld (de),a
inc hl
inc de
djnz fn_ext
fn4:
ret

;;------------------------------------------------
;; name of swap filename - MUST BE IN CAPITALS!
swap_filename:
defb "SWAP.DAT"
end_swap_filename:

;; number of records to read at a time during swap init
init_records_read:
defb 0

;; max number of records that can be read during swap init
num_records_init:
defb 0

;; set to 1 if swap.dat opened successfully.
swap_opened:
defb 0

;; points to story FCB
source_fcb:
defw 0

;; points to active FCB for reading
read_fcb:
defw 0

;; pointer to start of buffer for swap inits
buffer:
defw 0

;; FCB for swap.
swap_fcb:
defs 36
end_swap_fcb:

