;    Virtual memory read/write code for ZXZVM: Z-Code interpreter for the Z80 processor
;    Copyright (C) 2019  Kevin Thacker<kev@cpcfreak.co.uk>
;
;    This program is free software; you can redistribute it and/or modify
;    it under the terms of the GNU General Public License as published by
;    the Free Software Foundation; either version 2 of the License, or
;    (at your option) any later version.
;
;    This program is distributed in the hope that it will be useful,
;    but WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    GNU General Public License for more details.
;
;    You should have received a copy of the GNU General Public License
;    along with this program; if not, write to the Free Software
;    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; NOTE: None of this has been optimised

;;------------------------------------------------
;; Read from Z-Machine RAM or ROM
;; Enter:
;; DHL = virtual address
;; IX = page table
;; Exit:
;; A = byte, other registers corrupted
read_virtual:
push ix
call store_current_virt		;; store virtual memory address being accessed

call map_virtual_to_physical	;; is the virtual page mapped into physical ram?
jr nc,rv2

;; no, we need to map it.
pop ix
push ix
call cache_virtual

rv2:
call reset_page_age			;; reset the age of this page
pop ix
call update_page_age		;; update ages of all pages
ld a,(hl)
ret

;;------------------------------------------------
;; Write to Z-Machine RAM

;; Enter:
;; IX = page table
;; DHL = virtual address
;; A = byte
;; Exit:
;; registers corrupted.
write_virtual:

ld (virtual_data),a

ld a,d			;; do not allow write above 64KB
or a
ret nz

push ix
call store_current_virt			;; store virtual memory address being accessed

call map_virtual_to_physical		;; is the virtual page mapped into physical ram?
jr nc,wv2

;; no, we need to map it.
pop ix
push ix
call cache_virtual

wv2:
call reset_page_age				;; reset the age of this page
call set_page_dirty				;; mark as modified - will need commiting if discarded
pop ix
call update_page_age			;;;; update ages of all pages
ld a,(virtual_data)
ld (hl),a
ret


;;------------------------------------------------
;; Store virtual address
;; Entry:
;; DHL = virtual address
store_current_virt:
ld (virtual_address+0),hl
ld a,d
ld (virtual_address+2),a
ret

;;------------------------------------------------
;; Get virtual address

;; Exit:
;; DHL = virtual address
get_current_virt:
ld hl,(virtual_address+0)
ld a,(virtual_address+2)
ld d,a
ret

;;------------------------------------------------
;; Discard page, find an old page and load a new page in

cache_virtual:
;; find an old if USE_ADDR=1page and return it's page table location
call find_oldest_page
ret c


;; found an old page so commit existing one in the same page table location
call commit_page

;; set new virtual address for this page
call allocate_page

;; read virtual memory into physical
call restore_virtual_page

;; map virtual to physical
call virtual_to_physical
or a
ret
;;------------------------------------------------

set_page_dirty:
ld a,1
ld (ix+page_flags),a
ret

;;------------------------------------------------
;; Setup page data which defines the physical addresses available
;; and initial corresponding virtual pages

;; Enter:
;; HL = address of physical ram start to use for paging virtual memory into
;; DE = length of physical ram 
;; IX = page table
;; Will allocate pages for virtual memory use.
virt_init:
;; count number of pages we can cache at one time
push hl




ld l,e
ld h,d
ld d,0

ld bc,PAGE_SIZE_BYTES
virti00a:
or a
sbc hl,bc
;; zero
ld a,h
or l
jr z,virti00c

;; HL<DE?
bit 7,h
jr nz,virti00c
inc d
jr virti00a

virti00c:
ld (ix+0),d

vinit2:
pop hl

ld b,(ix+0)
inc b
dec b
ret z
push ix
inc ix
vinit1:
;; set physical page address
ld (ix+page_physical_address+0),l
ld (ix+page_physical_address+1),h
;; clear modified
ld (ix+page_flags),0
;; set initial age
ld (ix+page_age),MAX_PAGE_AGE
ld de,PAGE_SIZE_BYTES
add hl,de
ld de,page_data_size
add ix,de
djnz vinit1
pop ix
or a
ret


;;------------------------------------------------
;; for game restart, setup default virtual addresses and 
;; read pages in for those
;; Entry:
;; IX = page table

virt_reinit:
call virt_set_default
call virt_restore_init
ret

;;------------------------------------------------
;; set default virtual addresses - mapped to consecutive pages in z-machine ram

;; Entry:
;; IX = page table
virt_set_default:
ld b,(ix+0)
inc b
dec b
ret z

push ix
;; set virtual page address
ld hl,0
ld d,0
inc ix
vinit4:
push bc
ld (ix+page_virtual_address+0),l
ld (ix+page_virtual_address+1),h
ld (ix+page_virtual_address+2),d
ld bc,PAGE_SIZE_BYTES
add hl,bc
ld bc,page_data_size
add ix,bc
pop bc
djnz vinit4
pop ix
ret


;;------------------------------------------------
;; read initial virtual pages
;; Entry:
;; IX = page table

virt_restore_init:
ld b,(ix+0)
inc b
dec b
ret z
push ix
inc ix
vinit5:
push bc
call restore_virtual_page
ld bc,page_data_size
add ix,bc
pop bc
djnz vinit5
pop ix
ret


;;------------------------------------------------
;; If page is modified commit it back to swap
;;
;; IX = current page table entry

commit_page:
ld a,(ix+page_flags)
or a
ret z
xor a
ld (ix+page_flags),a
;; page is dirty
call store_virtual_page
ret

;;------------------------------------------------
;; Find oldest page
;;
;; Entry:
;; IX = page table
;; Exit:
;; carry false - found old page, IX=address of page in table
;; carry true - didn't find an old page. Error.

find_oldest_page:
ld b,(ix+0)
inc b
dec b
jr z,fop3
ld c,0   		;; will be max age
ld hl,0		;; will be address of page's data in page table

inc ix
fop:
;; page older?
ld a,(ix+page_age)
cp c
jr c,fop2
jr z,fop2
;; strictly older
ld c,a
push ix
pop hl
fop2:
ld de,page_data_size
add ix,de
djnz fop
;; did we find an old page?
ld a,h
or l
jr nz,fop4
;; error
fop3:
scf
ret

fop4:
;; success
push hl
pop ix
or a
ret



;;------------------------------------------------
;; Page align virtual address
;; Entry:
;; DHL = virtual address
;; Exit:
;; DHL = page aligned virtual address
align_virtual_page:
if PAGE_SIZE_BYTES=128
ld a,l
and %10000000
ld l,a
endif

if PAGE_SIZE_BYTES=256
ld l,0
endif

if PAGE_SIZE_BYTES=512
ld l,0
ld a,h
and &fe
ld h,a
endif

if PAGE_SIZE_BYTES=1024
ld l,0
ld a,h
and &fc
ld h,a
endif

if PAGE_SIZE_BYTES=2048
ld l,0
ld a,h
and &f8
ld h,a
endif

if PAGE_SIZE_BYTES=4096
ld l,0
ld a,h
and &f0
ld h,a
endif

if PAGE_SIZE_BYTES=8192
ld l,0
ld a,h
and &e0
ld h,a
endif

if PAGE_SIZE_BYTES=16384
ld l,0
ld a,h
and &c0
ld h,a
endif

ret

;;------------------------------------------------
;; allocate a page
;;
;; Enter:
;; ix = page table entry
;; dhl = virtual address (unaligned)
allocate_page:
call get_current_virt
call align_virtual_page
ld (ix+page_virtual_address+0),l
ld (ix+page_virtual_address+1),h
ld (ix+page_virtual_address+2),d
ret

;;------------------------------------------------
reset_page_age:
ld a,-1
ld (ix+page_age),a
ret

;;------------------------------------------------
;; updates the age of all pages
;; IX = page table
update_page_age:
ld b,(ix+0)
inc b
dec b
ret z
inc ix
ld de,page_data_size
upa:
;; increase age and cap at MAX_PAGE_AGE
ld a,(ix+page_age)
cp MAX_PAGE_AGE
jr z,upa2
inc a
ld (ix+page_age),a
upa2:
add ix,de
djnz upa
ret
;;------------------------------------------------
;; Find the physical address for a virtual page (if mapped)
;; This is slow linear search.
;;
;; Input:
;; DHL = virtual address
;; IX = page table
;; Output:
;; if virtual page is mapped: carry clear. HL = physical address, IX = page data address
;; if virtual page is not mapped: carry false. Registers corrupt.
map_virtual_to_physical:
ld b,(ix+0)
inc b
dec b
ret z
inc ix
call get_current_virt
call align_virtual_page
mvtop1:
ld a,l
cp (ix+page_virtual_address+0)
jr nz,mvtop2
ld a,h
cp (ix+page_virtual_address+1)
jr nz,mvtop2
ld a,d
cp (ix+page_virtual_address+2)
jr z,mvtop3
mvtop2:
ld a,d
ld de,page_data_size
add ix,de
ld d,a
djnz mvtop1
scf
ret

mvtop3:

;; success map virtual to physical
call virtual_to_physical
or a
ret

;;---------------------------------------------------------------------
;; Map virtual to physical
;; Entry:
;; IX = current page table entry
;; Exit:
;; HL = address in physical z80 ram corresponding to virtual address 
virtual_to_physical:
call get_current_virt	;; unaligned
;; calc relative to aligned virtual page address
ld a,d
sub (ix+page_virtual_address+2)
ld d,a
ld a,h
sbc a,(ix+page_virtual_address+1)
ld h,a
ld a,l
sbc a,(ix+page_virtual_address+0)
ld l,a
call get_page_physical_address
;; add on location in physical memory
add hl,de
ret

;;---------------------------------------------------------------------
;; IX = current page table entry
get_page_physical_address:
ld e,(ix+page_physical_address+0)
ld d,(ix+page_physical_address+1)
ret
;;---------------------------------------------------------------------
;; temp for storing data
virtual_data:
defb 0
virtual_address:
defs 3
